﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using UnityEngine;

namespace NovelInterpreter.Instance.Drawer
{
	public class WindowDrawer : IDrawer
	{
		/// <summary>
		/// 背景テクスチャ
		/// </summary>
		Texture2D texture = null;

		public Texture2D Texture
		{
			get { return this.texture; }
		}

		/// <summary>
		/// テクスチャ色
		/// </summary>
		public Color textureColor = new Color(1, 1, 1, 1);

		/// <summary>
		/// 背景色
		/// </summary>
		public Color backColor = new Color(0, 0, 0, 0);

		/// <summary>
		/// 前景色
		/// </summary>
		public Color frontColor = new Color(0, 0, 0, 0);

		/// <summary>
		/// 窓の横幅, 0の場合はテクスチャに依存
		/// </summary>
		public int width;

		/// <summary>
		/// 窓の縦幅, 0の場合はテクスチャに依存
		/// </summary>
		public int height;

		/// <summary>
		/// 真っ白なテクスチャ, 主に塗り重ねなどに利用
		/// </summary>
		Texture2D whiteTexture;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="texture">描画するテクスチャ</param>
		/// <param name="textureColor">テクスチャ色</param>
		/// <param name="backColor">背景色</param>
		/// <param name="frontColor">前景色</param>
		/// <param name="width">描画する横幅</param>
		/// <param name="height">描画する縦幅</param>
		public WindowDrawer(Texture2D texture, Color textureColor, Color backColor, Color frontColor, int width = 0, int height = 0)
		{
			this.whiteTexture = new Texture2D(1, 1, TextureFormat.RGBA32, false);
			this.whiteTexture.SetPixel(0, 0, Color.white);
			this.whiteTexture.Apply();

			if (texture == null) texture = this.whiteTexture;
			this.texture = texture;
			this.backColor = backColor;
			this.frontColor = frontColor;
			this.width = width;
			this.height = height;
		}

		/// <summary>
		/// ウィンドウの描画を行う
		/// </summary>
		/// <param name="transform">描画対象のTransform</param>
		public void Draw(Transform transform)
		{
			// 横幅と縦幅に入力があれば遷移する
			if (this.width > 0 && this.height > 0)
				Draw(transform, this.width, this.height);
			else
			{
				// 相対的な位置
				Rect rect = new Rect(
					transform.position.x * Screen.width,
					transform.position.y * Screen.height,
					transform.localScale.x * this.texture.width,
					transform.localScale.y * this.texture.height);

				DrawTextures(ref rect);
			}
		}

		/// <summary>
		/// ウィンドウの描画を行う, IDrawには実装されてないので注意
		/// </summary>
		/// <param name="transform">描画対象のTransform</param>
		/// <param name="width">描画したい横幅</param>
		/// <param name="height">描画したい縦幅</param>
		public void Draw(Transform transform, int width, int height)
		{
			// 直接幅などを指定
			Rect rect = new Rect(
				transform.position.x * Screen.width,
				transform.position.y * Screen.height,
				transform.localScale.x * width,
				transform.localScale.y * height);

			DrawTextures(ref rect);
		}

		/// <summary>
		/// ウィンドウの描画
		/// </summary>
		/// <param name="rect">描画範囲</param>
		void DrawTextures(ref Rect rect)
		{
			// 背景色を描画
			GUI.color = this.backColor;
			GUI.DrawTexture(rect, this.whiteTexture);	// ここで描画

			// テクスチャの参照があれば描画する
			if (this.texture != null)
			{
				GUI.color = this.textureColor;
				GUI.DrawTexture(rect, this.texture);
			}

			// このまま描画するとテクスチャを描画したときに
			// スケールが相対座標のままなので
			// 画像が予想外の拡大縮小をされてしまう
			GUI.color = this.frontColor;
			GUI.DrawTexture(rect, this.whiteTexture);
			GUI.color = Color.white;
		}
	}
}
